#include <ESPmDNS.h>
#include <WiFiUdp.h>
#include <WiFi.h>
#include <WiFiClient.h>
#include <ArduinoOTA.h>                         // Arduino OTA von Arduino, Juraj Andrassy
#include <Adafruit_NeoPixel.h>                  // Adafruit NeoPixel von Adafruit https://github.com/adafruit/Adafruit_NeoPixel

#include "DHT.h"                                // DHT sensor library von Adafruit https://github.com/adafruit/DHT-sensor-library
#include <MQTTClient.h>                         // MQTT von Joel Gaehwiller https://github.com/256dpi/arduino-mqtt

#include "config.h"                             // Config-Datei mit MQTT und WLAN-Zugang

void setupOTA();

const int Device_Online_FB_Address=1;						// Meldeadresse an die eine 1 gemeldet wird, sobald der µC online ist, wird automatisch über MQTT-Will auf 0 gesetzt sobald länger offline
const int Switch_FB_Address=2;                  // Meldeadresse an der Zustand des Schalters am SwitchPin gemeldet wird

const int VoltageReport_SD_Address=3;           // Magnetartikeladresse an die die Stellung des Potis gemeldet wird an A0, AD-Rohwert 0..1023
const int LED_SD_Address=4;                     // Magnetartikeladresse über die, die rote LED an/aus gemacht wird via Wert 0/1, wird über den 
                                                // Taster am ButtonPin umgeschaltet und die Magnetartikelstellungsänderung wird wiede an
                                                // WDP gemeldet
const int Restart_SD_Address=5;                 // Magnetartikeladresse über die ein Restart des µC ausgelöst wird
const int NeoPixel_SD_FirstAddress=101;         // Magnetartikeladresse ab der der Neopixel-Ring angesteuert wird
                                                // Adresse 101, 1. LED Rot-Wert 0..255
                                                // Adresse 102, 1. LED Grün-Wert 0..255
                                                // Adresse 103, 1. LED Blau-Wert 0..255
                                                // Adresse 104, 2. LED Rot-Wert 0..255
                                                // Adresse 105, 2. LED Grün-Wert 0..255
                                                // Adresse 106, 2. LED Blau-Wert 0..255
                                                // usw. 

const int PwSen_Address=10;                     // PwSen-Adresse an die gemeldet wird
                                                // Spannung in V gemessen welche am Poti eingestellt wird am Anschluss A0
                                                // simulierte Strom in mA zyklisch gesendet von 0..3000mA
                                                // Temperatur in °C gemessen via DHT22

#define DHTTYPE DHT22                           // DHT-Typ
#define DHTPIN 23                               // Port 23 Dateneingang Temperatur/Feuchtigkeitssensor DHT 22
#define HeartBeatPIN 2                          // Port 2 LED auf dem ESP32 Board, zeight ein Heart-Beat-Flacker-Signal an
#define MQTTPIN 33                              // Port 33 LED grün, leuchtet bei erfolgreicher Verbindung zum MQTT-Server
#define WIFIPIN 32                              // Port 32 LED gelb, leuchtet bei aktiver WLAN-Verbindng
#define ButtonPin 16                            // Port 16 Taster um LED_SD_Address umzuschalten 
#define NeoPIN 17                               // Port 17 Verbindung zum DIN-PIN des Neopixel
#define NUMPIXELS 12                            // Anzahl WS2812b LED auf dem Neopixel-Ring
#define LEDRed 18                               // Port 18 LED rot, angesteuert via LED_SD_Address
#define SwitchPin 19                            // Port 19 Schalter über den die Meldeadresse Switch_FB_Address angesteuert wird
#define PotiPin 35                              // Port 35 Analogeingang Poti

Adafruit_NeoPixel pixels(NUMPIXELS, NeoPIN, NEO_GRB + NEO_KHZ800);
#define DELAYVAL 500

unsigned long tick;

WiFiClient net;
MQTTClient mqtt;

int SwitchValueOld;
int ButtonValueOld;
int ADCValueOld;
unsigned long lmillis;
int PauseHeartBeat=0;
byte TickHeartBeat=0;
boolean up=true;
boolean updatePixels=false;
byte Fadearray[]={
     0,  2,  4,  8, 12, 20, 32, 44, 56, 68,
    80, 92,104,116,128,140,152,164,176,188,
   200,212,224,236,255,240,210,180,150,120,
    90, 60, 40, 25, 25, 40, 60, 90,120,150,
   180,210,240,255,230,205,180,155,130,105,
    80, 55, 30, 20, 16,  8,  4,  2,  0};

DHT dht(DHTPIN, DHTTYPE);

void connect();
void sendSD(int SDNum, int SDState){
  //Sende Magnetartikel-Stellung an WDP
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/SD/"+String(SDNum)+"/State", String(SDState));
}
void sendTemp(int PwSenNum, float temp){
  //Sende Temperatur in °C an WDP-Boostermanagement
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/PwSen/"+String(PwSenNum)+"/T_C", String(temp));
}
void sendCurrentA(int PwSenNum, float current){
  //Sende Strom in Ampere an WDP-Boostermanagement
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/PwSen/"+String(PwSenNum)+"/I_A", String(current));
}
void sendCurrentmA(int PwSenNum, float current){
  //Sende Strom in Milli-Ampere an WDP-Boostermanagement
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/PwSen/"+String(PwSenNum)+"/I_mA", String(current));
}
void sendVoltage(int PwSenNum, float voltage){
  //Sende Spannung in Volt an WDP-Boostermanagement
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/PwSen/"+String(PwSenNum)+"/U_V", String(voltage));
}
void sendFB(int FBNum, int FBState){
  //Sende Rückmelde-Status an WDP
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/FB/"+String(FBNum)+"/State", String(FBState));
}

void resetOldValues(){
  //Status-Werte erneut senden bzw. initialiases Senden
  SwitchValueOld=-10;
  ADCValueOld=-10;
  tick=0;
  ButtonValueOld=digitalRead(ButtonPin);
}

void statusLED(){
  if (mqtt.connected()){                             // Aktualisierung MQTT-Status LED
    digitalWrite(MQTTPIN, HIGH); 
  } else {
    digitalWrite(MQTTPIN, LOW); 
  }
  if (WiFi.waitForConnectResult() != WL_CONNECTED){  // Aktualisierung MQTT-Status LED
    digitalWrite(WIFIPIN, LOW); 
  } else {
    digitalWrite(WIFIPIN, HIGH); 
  }
}


void setup() {
  pinMode(MQTTPIN, OUTPUT);
  pinMode(WIFIPIN, OUTPUT);
  pinMode(HeartBeatPIN, OUTPUT);
  pinMode(LEDRed,OUTPUT);
  pinMode(ButtonPin,INPUT_PULLUP);
  pinMode(SwitchPin,INPUT_PULLUP);
  
  dht.begin();

  pixels.begin();
  pixels.clear();
  pixels.show();

  resetOldValues();

  Serial.begin(115200);
  Serial.println();
  Serial.println("Booting...");
  WiFi.hostname(WIFI_HOSTNAME); 
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, password);
  

  mqtt.begin(host, net);
  mqtt.setKeepAlive(60);
  statusLED();

  //Offline-Status via Rückmelder melden als LastWill via MQTT
  String s=String(wdpMQTTTopic)+"/Evt/FB/"+String(Device_Online_FB_Address)+"/State";
  mqtt.setWill(s.c_str(),"0",true,1);
  mqtt.onMessage(messageReceived);

  connect();
  Serial.println("Setup completed...");
}

void loop() {
  if (WiFi.status() == WL_CONNECTED) {
    ArduinoOTA.handle();
   }
    if ((tick % 100)==0){
      if (!mqtt.connected()) {
          connect();
        }
      
        mqtt.loop();
    }
    if (tick==0){
        //Temperatur-Sensor auslesen ~ alle 30 Sekunden

        float temp = dht.readTemperature();
        float humidity = dht.readHumidity();
      
        Serial.print("Reading temperatre... ");           // Kontrollausgabe der Werte via seriellem Interface
        if (!isnan(humidity) || !isnan(temp)) {
          sendTemp(PwSen_Address,temp);

          Serial.print("Temp: ");
          Serial.print(String(temp));
          Serial.print(" Humidity: ");
          Serial.print(String(humidity));
        }
        Serial.println("");

        
    }
    if ((tick % 100)==0){ // circa alle 100msec
      
      int val= !digitalRead(SwitchPin);                   //schalter auslesen und Status melden an Rückmelder Switch_FB_Address
      if (val!=SwitchValueOld){
        SwitchValueOld=val;
        Serial.println("SwitchPin Value: "+ val);
        sendFB(Switch_FB_Address,val);
      }
      
      val=analogRead(PotiPin);                            // Spannung an Poti an A0 auslesen und als Magnetartikelstellung an VoltageReport_SD_Address 
                                                          // melden und als PwSen-Spannungswert
      if (val!=ADCValueOld){
        ADCValueOld=val;
        Serial.println("ADC Value: "+ val);
        sendSD(VoltageReport_SD_Address,val);
        float voltage=val;
        voltage /= 4096;
        voltage *= 3.2;
        sendVoltage(PwSen_Address,voltage);
      }
      if (((tick % 200)==0) && updatePixels ){            // maximal alle 200 msec LED-Ring neu schreiben
        updatePixels=false;
        pixels.show();
      }

      val=digitalRead(ButtonPin);                         // Taster zum Umschalten der roten LED, 
                                                          // Umschaltung wird an Magnetartikelladresse LED_SD_Address gemeldet
      if (val!=ButtonValueOld){
        ButtonValueOld=val;
        
        if (val==LOW){
          digitalWrite(LEDRed, !digitalRead(LEDRed));
          sendSD(LED_SD_Address,digitalRead(LEDRed));
          Serial.println("Button pressed");
        }
        
      }
    }
    if ((tick % 1000)==0){                                  //circa alle 1000msec
        statusLED();
        sendCurrentmA(PwSen_Address,((int)(tick/1000))*100);  //Ausgabe eines Test Stromwert 0..3000mA in 100mA-Schritten

   }
    tick++;
    
    
  if (tick>=30000) tick=0;                                    // Tick-Counter zurücksetzen

  delay(1);
  //delay(30000);

  if (PauseHeartBeat<=0){                                     //Heart-Beat LED flackern lassen
    if (millis()-lmillis>=11){
      if (TickHeartBeat==58){
        PauseHeartBeat=550;
      } else {
        TickHeartBeat++;
        analogWrite(HeartBeatPIN,Fadearray[TickHeartBeat]); 
        lmillis=millis();
      }
      
    } else if (millis()<lmillis) { //wegen Überlauf
      lmillis=millis();
    }
  } else {
    PauseHeartBeat--;
    if (PauseHeartBeat==0){
      up=false;
      TickHeartBeat=0;
      TickHeartBeat++;
      analogWrite(HeartBeatPIN,Fadearray[TickHeartBeat]); 
      lmillis=millis();
    }
  }

}

void connect() {
  while(WiFi.waitForConnectResult() != WL_CONNECTED) {
    WiFi.begin(ssid, password);
    Serial.println("WiFi connection failed. Retry.");
  }

  if (WiFi.status() == WL_CONNECTED) {
    setupOTA();
  }
  
  Serial.print("Wifi connection successful - IP-Address: ");
  Serial.println(WiFi.localIP());
  statusLED();

  if (String(mqttUser).equals("") && String(mqttPassword).equals("")){
    while (!mqtt.connect(clientid)) {
      Serial.println("Wait for MQTT-Connect");
    }    
  } else if (String(mqttPassword).equals("")){
    while (!mqtt.connect(clientid, mqttUser)) {
      Serial.println("Wait for MQTT-Connect");
    }    
  } else {
    while (!mqtt.connect(clientid, mqttUser, mqttPassword)) {
      Serial.println("Wait for MQTT-Connect");
    }    
  }
  statusLED();


  Serial.println("MQTT connected!");
  

  mqtt.subscribe(String(wdpMQTTTopic)+"/#");    //alle MQTT-Nachrichten welche mit dem wdpMQTTTopic beginnen abonnieren

  //Online-Status melden an Device_Online_FB_Address
  mqtt.publish(String(wdpMQTTTopic)+"/Evt/FB/"+String(Device_Online_FB_Address)+"/State","1",true,1 );
  
}

String getValue(String data, char separator, int index)               // Trennzeichen-getrennten String zerlegen
{
    int found = 0;
    int strIndex[] = { 0, -1 };
    int maxIndex = data.length() - 1;

    for (int i = 0; i <= maxIndex && found <= index; i++) {
        if (data.charAt(i) == separator || i == maxIndex) {
            found++;
            strIndex[0] = strIndex[1] + 1;
            strIndex[1] = (i == maxIndex) ? i+1 : i;
        }
    }
    return found > index ? data.substring(strIndex[0], strIndex[1]) : "";
}




void processSDMessage(String &Subtopic, String &payload){
    // eingehenden Magnetartikelbefehl auswerten
    if (getValue(Subtopic,'/',1).equals("State")){
      //Magnetartikeladresse
      int num=getValue(Subtopic,'/',0).toInt();
      
      //Wert, wir lassen nur Ganzzahl-Werte im Bereich -32767..32767 zu...
      float f=payload.toFloat();
      if (f>32767) f=32767;
      if (f<-32767) f=-32767;
      int value=(int)(f+.5);
      
      if (num == LED_SD_Address){
        // rote LED angesteuert via LED_SD_Address
        digitalWrite(LEDRed, ((value!=0) ? HIGH : LOW));
      
      } else if ((num>=NeoPixel_SD_FirstAddress)&&(num<NeoPixel_SD_FirstAddress+NUMPIXELS*3)){
          //LED-Ring angsteuert via NeoPixel_SD_FirstAddress ff. Adresszuteilung s.o.
          int i=(num-NeoPixel_SD_FirstAddress)/3;
          int c=(num-NeoPixel_SD_FirstAddress)%3;
          if (value<0) value=0;
          u_int32_t color=pixels.getPixelColor(i);
          u_int8_t red=(color >> 16) & 0xFF;
          u_int8_t green=(color >> 8) & 0xFF;
          u_int8_t blue=(color >> 0) & 0xFF;

          switch (c) {
            case 0:
              red=value & 0xFF;
              break;
            case 1:
              green=value & 0xFF;
              break;
            case 2:
              blue=value & 0xFF;
              break;
          }
          
          pixels.setPixelColor(i, pixels.Color(red, green, blue));
          updatePixels=true;

      } else if (num == Restart_SD_Address){
        //ferngesteuerter Neustart des µC
        digitalWrite(MQTTPIN, LOW);
        digitalWrite(WIFIPIN, LOW);
        ESP.restart();
      }

    }
}


void messageReceived(String &topic, String &payload) {
  //eingehende MQTT-Nachricht auswerten, Testausgabe an seriellen Schnittstelle
  Serial.print("incoming: ");
  Serial.print(topic);
  Serial.print(" - ");
  Serial.print(payload);
  Serial.println();

  // Magnetartikel-Kommandos erkennen
  String s=String(wdpMQTTTopic)+"/Cmd/SD/";
  if (topic.startsWith(s)){
    s=topic.substring(s.length());
    processSDMessage(s,payload);
  }

  // Start einer WDP-Instanzerkennen -> neue InitTime-Meldung -> alle Werte erneut melden
  s=String(wdpMQTTTopic)+"/InitTime";
  if (topic.startsWith(s)){
    resetOldValues();
  }
  
}

//vorgefertige Routine für Update des Arduino Over the Air. Passwort siehe config.h
void setupOTA(){
  // Port defaults to 8266
  // ArduinoOTA.setPort(8266);

  // Hostname defaults to esp3232-[MAC]
  ArduinoOTA.setHostname(WIFI_HOSTNAME);

  // No authentication by default
  ArduinoOTA.setPassword(OTA_PASSWORD);

  // Password can be set with it's md5 value as well
  // MD5(admin) = 21232f297a57a5a743894a0e4a801fc3
  // ArduinoOTA.setPasswordHash("21232f297a57a5a743894a0e4a801fc3");

  ArduinoOTA.onStart([]() {
    String type;
    if (ArduinoOTA.getCommand() == U_FLASH) {
      type = "sketch";
    } else { // U_FS
      type = "filesystem";
    }

    // NOTE: if updating FS this would be the place to unmount FS using FS.end()
    Serial.println("Start updating " + type);
  });
  ArduinoOTA.onEnd([]() {
    Serial.println("\nEnd");
  });
  ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
    Serial.printf("Progress: %u%%\r", (progress / (total / 100)));
  });
  ArduinoOTA.onError([](ota_error_t error) {
    Serial.printf("Error[%u]: ", error);
    if (error == OTA_AUTH_ERROR) {
      Serial.println("Auth Failed");
    } else if (error == OTA_BEGIN_ERROR) {
      Serial.println("Begin Failed");
    } else if (error == OTA_CONNECT_ERROR) {
      Serial.println("Connect Failed");
    } else if (error == OTA_RECEIVE_ERROR) {
      Serial.println("Receive Failed");
    } else if (error == OTA_END_ERROR) {
      Serial.println("End Failed");
    }
  });
  ArduinoOTA.begin();
  
}
